<?php
require_once __DIR__ . '/db.php';

function h($v): string { return htmlspecialchars((string)$v, ENT_QUOTES, 'UTF-8'); }

function json_out($data, int $code = 200): void {
  http_response_code($code);
  header('Content-Type: application/json; charset=utf-8');
  echo json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
  exit;
}

function read_json(): array {
  $raw = file_get_contents('php://input');
  if (!$raw) return [];
  $data = json_decode($raw, true);
  return is_array($data) ? $data : [];
}

function normalize_phone(string $phone): string {
  $p = preg_replace('/\D+/', '', $phone);
  if ($p === '') return '';
  // Handle common KE formats: 07.., 7.., 2547..
  if (str_starts_with($p, '0') && strlen($p) === 10) $p = '254' . substr($p, 1);
  if (str_starts_with($p, '7') && strlen($p) === 9)  $p = '254' . $p;
  if (str_starts_with($p, '254') && strlen($p) === 12) return $p;
  return $p; // fallback (do not reject; validate later)
}

function api_key_ok(): bool {
  $cfg = app_cfg();
  $key = $cfg['security']['api_key'] ?? '';
  $hdr = $_SERVER['HTTP_X_API_KEY'] ?? '';
  $q   = $_GET['api_key'] ?? ($_POST['api_key'] ?? '');
  $got = $hdr ?: $q;
  return is_string($got) && $got !== '' && hash_equals($key, $got);
}

function require_api_key(): void {
  if (!api_key_ok()) json_out(['ok'=>false,'error'=>'Unauthorized'], 401);
}

function start_session(): void {
  $cfg = app_cfg();
  if (session_status() !== PHP_SESSION_ACTIVE) {
    session_name($cfg['security']['session_name'] ?? 'SPLOYSESS');
    session_start();
  }
}

function csrf_token(): string {
  start_session();
  $cfg = app_cfg();
  $k = $cfg['security']['csrf_key'] ?? 'x';
  if (empty($_SESSION['_csrf'])) {
    $_SESSION['_csrf'] = hash('sha256', random_bytes(32) . $k);
  }
  return $_SESSION['_csrf'];
}

function csrf_check(): void {
  start_session();
  $t = $_POST['_csrf'] ?? '';
  if (!is_string($t) || $t === '' || empty($_SESSION['_csrf']) || !hash_equals($_SESSION['_csrf'], $t)) {
    http_response_code(403);
    echo "CSRF blocked.";
    exit;
  }
}

function require_admin(): void {
  start_session();
  if (empty($_SESSION['admin_user'])) {
    header("Location: " . base_url() . "/admin/login.php");
    exit;
  }
}

function base_url(): string {
  $cfg = app_cfg();
  $b = rtrim((string)($cfg['base_url'] ?? ''), '/');
  return $b;
}

function flash_set(string $msg, string $type='info'): void {
  start_session();
  $_SESSION['_flash'] = ['msg'=>$msg,'type'=>$type];
}

function flash_get(): ?array {
  start_session();
  if (empty($_SESSION['_flash'])) return null;
  $f = $_SESSION['_flash'];
  unset($_SESSION['_flash']);
  return $f;
}
