<?php
require_once __DIR__ . '/../../includes/helpers.php';
require_api_key();

$data = read_json();

$phone      = normalize_phone((string)($data['phone'] ?? ''));
$pos_txn_no  = trim((string)($data['pos_txn_no'] ?? ''));
$branch_code = trim((string)($data['branch_code'] ?? ''));
$cashier     = trim((string)($data['cashier'] ?? ''));
$sale_total  = (float)($data['sale_total'] ?? 0);

if ($phone === '' || strlen($phone) < 9) json_out(['ok'=>false,'error'=>'Invalid phone'], 422);
if ($pos_txn_no === '') json_out(['ok'=>false,'error'=>'Missing pos_txn_no'], 422);
if ($sale_total <= 0) json_out(['ok'=>false,'error'=>'Invalid sale_total'], 422);

$db = db();

// Find active rule
$rule = $db->query("SELECT * FROM loyalty_rules WHERE is_active=1 ORDER BY id DESC LIMIT 1")->fetch_assoc();
$ppk = (float)($rule['points_per_kes'] ?? 0.02);
$min = (float)($rule['min_spend'] ?? 0);
$max = $rule['max_points_per_sale'] ?? null;

if ($sale_total < $min) {
  json_out(['ok'=>true,'awarded'=>0,'reason'=>"Below min spend (KES " . number_format($min,2) . ")"]);
}

$points = (int)floor($sale_total * $ppk);
if ($points < 0) $points = 0;
if ($max !== null && $max !== '' && (int)$max > 0) $points = min($points, (int)$max);

if ($points <= 0) json_out(['ok'=>true,'awarded'=>0,'reason'=>'No points computed']);

// Prevent double-award for same receipt
$stmt = $db->prepare("SELECT id FROM loyalty_transactions WHERE pos_txn_no=? LIMIT 1");
$stmt->bind_param("s", $pos_txn_no);
$stmt->execute();
$exists = $stmt->get_result()->fetch_assoc();
$stmt->close();
if ($exists) json_out(['ok'=>false,'error'=>'This POS receipt already awarded loyalty'], 409);

$db->begin_transaction();
try {
  // Create customer if not exists
  $stmt = $db->prepare("SELECT id, name FROM loyalty_customers WHERE phone=? LIMIT 1");
  $stmt->bind_param("s", $phone);
  $stmt->execute();
  $cust = $stmt->get_result()->fetch_assoc();
  $stmt->close();

  if (!$cust) {
    $name = trim((string)($data['name'] ?? ''));
    $ins = $db->prepare("INSERT INTO loyalty_customers (phone, name) VALUES (?, ?)");
    $ins->bind_param("ss", $phone, $name);
    $ins->execute();
    $customer_id = (int)$ins->insert_id;
    $ins->close();

    $acc = $db->prepare("INSERT INTO loyalty_accounts (customer_id, tier, points_balance, lifetime_points) VALUES (?, 'Silver', 0, 0)");
    $acc->bind_param("i", $customer_id);
    $acc->execute();
    $acc->close();
    $cust_name = $name;
  } else {
    $customer_id = (int)$cust['id'];
    $cust_name = (string)($cust['name'] ?? '');
    // Ensure account exists
    $db->query("INSERT IGNORE INTO loyalty_accounts (customer_id, tier, points_balance, lifetime_points) VALUES ($customer_id, 'Silver', 0, 0)");
  }

  $meta = [
    'rule_id' => (int)($rule['id'] ?? 0),
    'rule_note' => (string)($rule['note'] ?? ''),
    'raw' => $data,
  ];
  $meta_json = json_encode($meta, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

  $ins = $db->prepare("
    INSERT INTO loyalty_transactions
    (customer_id, branch_code, cashier, pos_txn_no, sale_total, points_awarded, source, meta_json)
    VALUES (?, ?, ?, ?, ?, ?, 'RMS', ?)
  ");
  $ins->bind_param("isssdiss", $customer_id, $branch_code, $cashier, $pos_txn_no, $sale_total, $points, $meta_json);
  $ins->execute();
  $loy_txn_id = (int)$ins->insert_id;
  $ins->close();

  $upd = $db->prepare("
    UPDATE loyalty_accounts
    SET points_balance = points_balance + ?, lifetime_points = lifetime_points + ?
    WHERE customer_id = ?
  ");
  $upd->bind_param("iii", $points, $points, $customer_id);
  $upd->execute();
  $upd->close();

  $balRow = $db->query("SELECT tier, points_balance FROM loyalty_accounts WHERE customer_id=$customer_id")->fetch_assoc();
  $balance = (int)($balRow['points_balance'] ?? 0);
  $tier = (string)($balRow['tier'] ?? 'Silver');

  // Save note mapping for RMS receipts (optional)
  $note = "LOYALTY:$phone|PTS:+$points|BAL:$balance|LTX:$loy_txn_id";
  $stmt = $db->prepare("INSERT INTO pos_receipt_notes (pos_txn_no, note) VALUES (?, ?) ON DUPLICATE KEY UPDATE note=VALUES(note)");
  $stmt->bind_param("ss", $pos_txn_no, $note);
  $stmt->execute();
  $stmt->close();

  $db->commit();

  json_out([
    'ok'=>true,
    'customer'=>[
      'id'=>$customer_id,
      'phone'=>$phone,
      'name'=>$cust_name,
      'tier'=>$tier,
    ],
    'awarded'=>$points,
    'balance'=>$balance,
    'loyalty_txn_id'=>$loy_txn_id,
    'receipt_note'=>$note,
  ]);
} catch (Throwable $e) {
  $db->rollback();
  json_out(['ok'=>false,'error'=>'Award failed'], 500);
}
